
 ; reads steering wheel control voltage and sends IR code 

; AN3 for SWC in
; 36/38/40kHz modulation output at pin 5
; demodulated output at pin 7 (GP0)
; hysteresis AN1
; program switch GP5
; IR sensor GP3

	ERRORLEVEL -302
	ERRORLEVEL -306

	    list      p=12F617        	; list directive to define processor
     #include <p12F617.inc>        ; processor specific variable definitions


   __CONFIG   _CP_OFF & _BOR_OFF & _MCLRE_OFF & _WDT_OFF & _PWRTE_ON &_INTRC_OSC_NOCLKOUT & _IOSCFS_8MHZ &_WRT_OFF

; RAM 
msCOUNT_LS	equ	H'20'	; 110ms counter ls byte
msCOUNT_MS	equ	H'21'	; 110ms counter ms byte
INT_FLG			equ	H'22'	;  flag to indicate 50us interrupt
STORE1			equ	H'23'	; delay counter	
STORE2			equ	H'24'	; delay counter
SWITCH			equ	H'25'	; switch flag
TEMP			equ	H'26'	; temporary
AD_STORE		equ	H'27'	; A/D storage
SHOW			equ	H'28'	; show protocol (flash LED2)
T1_STO_H		equ	H'29'	; timer1 store ms byte
T1_STO_L		equ	H'2A'	; timer1 store ls byte	
TEMPM			equ	H'2B'	; temp ms byte
TEMPL			equ	H'2C'	; temp ls byte
TEMP2M		equ	H'2D'	; temp2 ms byte
TEMP2L			equ	H'2E'	; temp2 ls byte
RPT				equ	H'2F'	; repeat flag (for switch 10 that has no repeat)
HYSTERESIS	equ	H'31'	; hysteresis read from VR1 (0-500mV range for 0-5V measured at TP1)
SW_VAL		equ	H'32'	; programming/ switch value

; DATA ( 4 blocks for writing/reading) data in the ls bytes only. Progtrammed to H0700 to H0717
PROTOCOLms		equ	H'40'	; 
PROTOCOLls		equ	H'41'	; Protocol is least significant 4-bits 0 = NEC, 1 = Sony;  2 = RC5. Plus for Sony, the address length is in ms 4-bits 0= 5-bits, 1 = 8 bits, 2 = 13bits
ADDRESSxms		equ	H'42'	; 
ADDRESSxls		equ	H'43'	; IR code address used as ms byte for Sony 20bit data (13bit address) ie 5 ms bits and NEC address extension
ADDRESSyms		equ	H'44'
ADDRESSyls		equ	H'45'	; IR code address	ls byte
; add memory space	H'46' unused
; add memory space	H'47' unused
; end 4-word block

; switch 1
SWC1ms			equ	H'48'		; 
SWC1ls				equ	H'49'	; switch voltage 1 (H0704 in flash)
COMMAND1ms		equ	H'4A'	; 
COMMAND1ls		equ	H'4B'	; IR code command 1
; switch 2
SWC2ms			equ	H'4C'	; 
SWC2ls				equ	H'4D'	; switch voltage 2
COMMAND2ms		equ	H'4E'	; 
COMMAND2ls		equ	H'4F'	; IR code command 2
; end 4-word block

;switch 3
SWC3ms			equ	H'50'	; 
SWC3ls				equ	H'51'	; switch voltage 3
COMMAND3ms		equ	H'52'	; 
COMMAND3ls		equ	H'53'	; IR code command 3
; switch 4
SWC4ms			equ	H'54'	; 
SWC4ls				equ	H'55'	; switch voltage 4
COMMAND4ms		equ	H'56'	; 
COMMAND4ls		equ	H'57'	; IR code command 4
; end 4-word block

; switch 5
SWC5ms			equ	H'58'	; 
SWC5ls				equ	H'59'	; switch voltage 5
COMMAND5ms		equ	H'5A'	; 
COMMAND5ls		equ	H'5B'	; IR code command 5
;switch 6
SWC6ms			equ	H'5C'	; 
SWC6ls				equ	H'5D'	; switch voltage 6
COMMAND6ms		equ	H'5E'	; 
COMMAND6ls		equ	H'5F'	; IR code command 6
; end 4-word block
; end 16 word
;........................................................................................................................................
; switch 7
SWC7ms			equ	H'60'	; 
SWC7ls				equ	H'61'	; switch voltage 7
COMMAND7ms		equ	H'62'	; 
COMMAND7ls		equ	H'63'	; IR code command 7
; switch 8
SWC8ms			equ	H'64'	; 
SWC8ls				equ	H'65'	; switch voltage 8
COMMAND8ms		equ	H'66'	; 
COMMAND8ls		equ	H'67'	; IR code command 8
; end 4-word block

;switch 9
SWC9ms			equ	H'68'	; 
SWC9ls				equ	H'69'	; switch voltage 9
COMMAND9ms		equ	H'6A'	; 
COMMAND9ls		equ	H'6B'	; IR code command 9
;switch 10
SWC10ms			equ	H'6C'	; 
SWC10ls			equ	H'6D'	; switch voltage 10
COMMAND10ms		equ	H'6E'	; 
COMMAND10ls		equ	H'6F'	; IR code command 10
; end 4-word block. 24 words total

; H0718 to H071F are also programmed with all banks RAM to make up the 32 words programmed, but are not used.

; All banks RAM
W_TMP			equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 

; IR remote control decoding
REMOTE_A		equ	H'72'	; remote control Address byte
REM_A_BAR	equ	H'73'	; complementary value of address (NEC) or address extension
REM_A_MS		equ	H'74'	; ms byte of address (Sony) if 20bit format (13bit address)
REMOTE_C		equ	H'75'	; remote control Command byte
REM_C_BAR	equ	H'76'	; complementary value of command (NEC)
BIT_COUNT		equ	H'77'	; bit counter
TOGGLE		equ	H'78'	; toggle bit (RC5)
SONY_ADD_L	equ	H'79'	; sony address length 5, 8 and 13 bits (for 12, 15 and 20 bits total when including command bits (7))
COUNTER		equ	H'7A'	; counter for period
DATA_ADD		equ	H'7B'	; Load initial data address when writing to memory
READADDMS	equ	H'7C'	; flash memory read address ms byte making up a 14 bit byte (ms byte has 6 bits)
READADDLS	equ	H'7D'  	; flash memory read address ls byte (ls byte has 8 bits)
;___________________________________________________________
; initial values

;	org	H'700'		; start address of where data memory is stored for PROTOCOLms, PROTOCOLls and following

; ******************************************************************

; start at memory 0

	org		0				; reset vector
	goto	MAIN
; interrupt vector
	org		4
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0

; clear timer0 flag 
	bcf		INTCON,TMR0IF
; reload for 50us
	movlw	H'D4'			; 50 - 255 
	movwf	TMR0
	bsf		INT_FLG,0		; set flag to indicate a 50us period interrupt

; increase repeat counter used for IR coding repeat function counts at 50us rate
	incfsz	msCOUNT_LS,f	; increase counter ls byte
	goto	RECLAIM
	incf		msCOUNT_MS,f	; increase counter ms byte

; end of interrupt reclaim w and status 
RECLAIM
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf  	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

; ***********************************************************
		
MAIN
;  set oscillator calibration
	bsf		STATUS,RP0     	; bank 1
        movlw   D'0'     			 ; set oscillator to factory calibrated frequency 
        movwf   OSCTUNE
	bcf		STATUS,RP0
SETUP
; set inputs/outputs
	movlw	B'00000000'
	movwf	GPIO			; ports low
	movlw	B'00000111'		; comparators off
	movwf	CMCON0
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00100000'		; pullups off/on
	movwf	WPU
	movlw	B'00111010'		; outputs/inputs set 
	movwf	TRISIO			; port data direction register
	movlw	B'00000000'		; settings (pullups) prescaler/2
	movwf	OPTION_REG
; analog inputs, A/D
	movlw	B'00101010'
	movwf	ANSEL			; digital I/O and analog at AN1 and AN3
	bcf		STATUS,RP0	; select memory bank 0
; bits 4-2
; 001 = Channel 01 (AN1)
; 011 = Channel 03 (AN3)
	movlw	B'00000100'		; channel 1, left justified, VDD ref etc
	movwf	ADCON0
	bsf		ADCON0,0		; A/D on

; pwm set
	bsf		STATUS,RP0	; select memory bank 1
	movlw	D'52'			; 
	movwf	PR2				; PWM period register. Use D52  for 37.74kHz NEC protocol, D49 for 40kHz Sony protocol, 55 for RC5 35.71kHz
	bcf		STATUS,RP0	; memory bank 0
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	movlw	B'00000000'		; prescaler /1 
	movwf	T2CON
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'		; set PWM mode
	movwf	CCP1CON		; enable PWM operation

;Timer1 on
	movlw 	B'00010001'		; divide by 2 for 1us per count
	movwf	T1CON
; start delay for S1 closure detect
	call		DELAY100		;  100ms	

; Start by a read of stored data
; retrieve IR command codes for up to 10 switches
; common info: IR Protocol and address

	movlw	H'41'
	movwf	FSR			; indirect pointer address

	movlw	H'07'			; ms address byte (H0700 start)
	movwf	READADDMS 
	movlw	H'00'
	movwf	READADDLS	; ls address byte
READ_LOOP
	call		READ			; 'w' has data
	movwf	INDF			; placed in RAM at FSR address

	incf		READADDLS,f	; next ls address byte
; stop when FSR=6F
	movf	FSR,w
	sublw	H'6E'
	btfss	STATUS,C		; when carry is clear then end of memory to read
	goto	EN
	incf		FSR,f
	incf		FSR,f			; write to RAM on ls bytes only (ms byte of 14-bit data ignored) 
	goto	READ_LOOP

; Interrupts
; enable interrupts
EN
	bsf		INTCON,T0IE		; timer 0 interrupt
	bcf		INTCON,T0IF
	bsf 		INTCON,GIE		; set global interrupt enable 

CHECK_S1
; check if S1 closed or open
	btfsc	GPIO,5			; if high run normal mode 
	goto	START_RUN
; recheck after delay (allows 100nF capacitor to charge via weak pullup)
	call		DELAY100		;  100ms	
	btfsc	GPIO,5			; if high run normal mode
	goto	START_RUN
; recheck after delay
	call		DELAY100		;  100ms	
	btfsc	GPIO,5			; if low run the program mode
	goto	START_RUN
	
PROGRAM_MODE
; PROGRAM mode where IR code is read and referenced against the SWC voltage when switch is pressed. Switch counter increased (counting from 1-10 switches)
; clear switch value
	clrf		SW_VAL		; counts up to 9. Counts up each time S1 pressed to record code

; clear all stored codes to FF
; clear RAM
	movlw	H'40'
	movwf	FSR			; indirect pointer address
LOAD_FF
	movlw	H'FF'			; set at maximum value
	movwf	INDF			; clear memory
; stop when FSR=6F
	movf	FSR,w
	sublw	H'6E'
	btfss	STATUS,C		; when carry is clear then end of memory to read
	goto	WRITE_CLR
	incf		FSR,f			; write to RAM on ls bytes only (ms byte of 14-bit data ignored) 
	goto	LOAD_FF

WRITE_CLR
; clear flash in two 16byte steps
; first 16 bytes H0700-H070F
	movlw	H'07'
	movwf	READADDMS	 ; MS Byte of Program Address to write
	movlw	H'00'	
	movwf	READADDLS 	; LS Byte of Program Address to write
	movlw	H'40'			; ram start
	movwf	DATA_ADD		; Load initial data address
	call		WRITE			; write to flash memory
; second 16 bytes H0710-H071F
	movlw	H'07'
	movwf	READADDMS	 ; MS Byte of Program Address to write
	movlw	H'10'	
	movwf	READADDLS 	; LS Byte of Program Address to write
	movlw	H'60'			; ram start
	movwf	DATA_ADD		; Load initial data address
	call		WRITE			; write to flash memory

; wait for S1 open 
S1_OPEN;?
	call		DELAY100		;  100ms	
	btfss	GPIO,5			; if high continue with program mode
	goto	S1_OPEN
; flash LED for switch value (1-10 times) SW_VAL counts up from 0 to 9
	movf	SW_VAL,w
	movwf	TEMP			; working value
F1	bsf		GPIO,0			; LED on
	call		DELAY100		;  100ms	
	call		DELAY100		;  100ms	
	call		DELAY100		;  100ms	
	bcf		GPIO,0			; LED off
	call		DELAY100		;  100ms	
	call		DELAY100		;  100ms	
	call		DELAY100		;  100ms	

	movf	TEMP,w
	btfsc	STATUS,Z		; when zero end flash sequence 
	goto	END_FLSH
	decf	TEMP,f
	goto	F1

END_FLSH
; wait for valid IR code reception
IR_IN
	call 		START			;  if returns with 00 or FF then not valid. (A0 for 'A o'k is valid code)

;  compare 'w' against H'A0'
	xorlw	H'A0'
	btfsc	STATUS,Z
	goto	VALID			; valid code so store

; check if S1 pressed
	btfsc	GPIO,5	
	goto	IR_IN			; wait again, not valid
	goto	S1_O			; bypass programming this switch move on to next one

VALID
; code is valid
; light up LED2. LED2 lights when valid code recorded
	movlw	D'52'			; PWM fully on
	movwf	CCPR1L		; ms byte of PWM
	clrf		SHOW

; show protocol. Continuous=NEC, Momentary flash off=Sony, Two momentary flash off=RC5
; wait for S1 to close
S1_C
	call		DELAY100		;  100ms	
	btfsc	SHOW,0
	goto	TEST_SW		; read switch closure
; only run this once	
	bsf		SHOW,0			; set so run flash once
	call		DELAY100		;  100ms	
	call		DELAY100
	movf	PROTOCOLls,w
	andlw	B'00001111'		; only ls bits
	movwf	TEMP
	xorlw	D'1'				; sony
	btfsc	STATUS,Z
	goto	FLASH_OFF1
	movf	TEMP,w
	xorlw	D'2'				; RC5
	btfsc	STATUS,Z
	goto	FLASH_OFF2

TEST_SW
	btfsc	GPIO,5
	goto	S1_C
	goto	SET_READ

FLASH_OFF2
	clrf		CCPR1L		; LED off
	call		DELAY100
	movlw	D'52'			; PWM fully on so LED2 on
	movwf	CCPR1L		; ms byte of PWM
	call		DELAY100
FLASH_OFF1
	clrf		CCPR1L		; LED off
	call		DELAY100
	movlw	D'52'			; PWM fully on so LED2 on
	movwf	CCPR1L		; ms byte of PWM
	call		DELAY100

	goto	S1_C

; when S1 pressed, stores both IR code and address and the SWC voltage for the current switch number

SET_READ
; setup for SWC input read
; bits 4-2
; 011 = Channel 03 (AN3)
	movlw	B'00001100'		; channel 3, left justified, VDD ref etc
	movwf	ADCON0
	bsf		ADCON0,0		; A/D on
; delay	
	call		DELAY100		;  100ms	; setup for SWC input read
	call		ACQUIRE_AD	; read switched input
	
; load RAM address for SWC voltage
; multiply SW_VAL by 4 (as each switch voltage storage is 4 apart)
	movf	SW_VAL,w		; 
	movwf	TEMP
	bcf		STATUS,C
	rlf		TEMP,f
	bcf		STATUS,C
	rlf		TEMP,w
	addlw	H'49'			; SWC for switch 1. find SWC storage for the switch selected
	movwf	FSR			; pointer to address
	movf	ADRESH,w		; SWC value
	movwf	INDF			; place in SWC(X)ls where X is that switches SWC address 
	incf		FSR,f
	incf		FSR,f			; add two for command byte
; common files	
	movf	REMOTE_C,w
	movwf	INDF			; COMMAND(X)ls
	movf	REMOTE_A,w
	movwf	ADDRESSyls	; address ls byte ADDRESSyls

; check protocol
	movf	PROTOCOLls,w
	andlw	B'00001111'		; just ls bits
	btfsc	STATUS,Z
	goto	NEC2
	xorlw	B'00000001'
	btfsc	STATUS,Z
	goto	SONY2
RC5_2
	goto	WRTE_FLSH

SONY2
; if 13 bit address then bit 5 is set
	btfss	PROTOCOLls,5		; set for 13bit address	
	goto	SONY_OTHER
;  then load REMA_MS to ADDRESSxls
	movf	REM_A_MS,w
	movwf	 ADDRESSxls
SONY_OTHER
	goto	WRTE_FLSH

NEC2
	movf	REM_A_BAR,w
	movwf	ADDRESSxls		; 

WRTE_FLSH
; write to flash in two 16byte steps
; first 16 bytes H0700-H070F
	movlw	H'07'
	movwf	READADDMS	 ; MS Byte of Program Address to write
	movlw	H'00'	
	movwf	READADDLS 	; LS Byte of Program Address to write
	movlw	H'40'			; ram start
	movwf	DATA_ADD		; Load initial data address
	call		WRITE			; write to flash memory
; second 16 bytes H0710-H071F
	movlw	H'07'
	movwf	READADDMS	 ; MS Byte of Program Address to write
	movlw	H'10'	
	movwf	READADDLS 	; LS Byte of Program Address to write
	movlw	H'60'			; ram start
	movwf	DATA_ADD		; Load initial data address
	call		WRITE	

; wait for S1 open
S1_O
	call		DELAY100		;  100ms	
	btfss	GPIO,5
	goto	S1_O

; LED2 out when stored
	clrf		CCPR1L		; cleared for LED off, ms byte of PWM

; SW_VAL increased; stops at end of 9
	incf		SW_VAL,f
	movf	SW_VAL,w
	sublw	D'09'			; if over 9 end of programming. 
	btfsc	STATUS,C
	goto	S1_OPEN		; next switch

STOP_AT_END
	goto	STOP_AT_END	; loop until powered off	

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
	
; when switch is not pressed at startup, read hysteresis and SWC input

START_RUN
	call		HYSTERESIS_RUN	; get hysteresis

; read AN3 input
LOOP
READ_CYCLE
; check if S1 closed or open
	btfss	GPIO,5			; if low get hysteresis so pressing S1 enables adjustment of hysteresis
	call		HYSTERESIS_RUN

; read SWC input
	movf	ADRESH,w

	movwf	AD_STORE		; last read storage
; if switch closed already, check with A/D again but no delay
	btfss	SWITCH,0		; bypass delay when switch closed and first IR code sent
	call		DELAY100		; make sure 
	call		ACQUIRE_AD	; read switched input

; is new A/D reading within plus or minus two values window

; add  3 to AD_STORE
	movlw	D'3'
	addwf	AD_STORE,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< last stored A/D plus window so reading is within positive range
	goto	NOT_IN_RANGE		; ADRESH >  range	

; take 2 from AD_STORE
	movlw	D'2'
	subwf	AD_STORE,w
	btfss	STATUS,C
	clrw					; not below 0
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> last stored setting so reading is within negative range
	goto	TEST_MAX		; within range

NOT_IN_RANGE
; not within range so not matching
	clrf		SWITCH			; no match so clear closed switch flag
	clrf		RPT				; code repeat flag
	goto	READ_CYCLE	; two readings differ so test again

TEST_MAX
; if at maximum (FF) including hysteresis range below maximum, ignore
	comf	HYSTERESIS,w	; get hysteresis below FF
	movwf	TEMP
	movlw	D'02'			; add safety margin to extend hysteresis by 2
	subwf	TEMP,w
	
	subwf	ADRESH,w		; switch reading
	btfsc	STATUS,C
	goto	SW0			; no switch

; compare against each switch voltage setting adding and subtracting hysteresis

CK_SW1	; SWC1ls	; switch at voltage 1
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC1ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW2		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC1ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW1			; switch found
		
CK_SW2	; SWC2ls	; switch at voltage 2
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC2ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW3		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC2ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW2			; switch found

CK_SW3	; SWC3ls	; switch at voltage 3
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC3ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW4		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC3ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW3			; switch found

CK_SW4	; SWC4ls	; switch at voltage4
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC4ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW5		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC4ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW4			; switch found

CK_SW5	; SWC5ls	; switch at voltage5
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC5ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW6		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC5ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW5			; switch found

CK_SW6	; SWC6ls	; switch at voltage6
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC6ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW7		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC6ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW6			; switch found

CK_SW7	; SWC7ls	; switch at voltage7
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC7ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW8		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC7ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW7			; switch found

CK_SW8	; SWC8ls	; switch at voltage8
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC8ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW9		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC8ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW8			; switch found

CK_SW9	; SWC9ls	; switch at voltage9
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC9ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	CK_SW10		; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC9ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW9			; switch found

CK_SW10	; SWC10ls	; switch at voltage10
; window of +/-hysteresis
; add  hysteresis to SWC(x)ls 
	movf	HYSTERESIS,w
	addwf	SWC10ls,w
	btfsc	STATUS,C
	movlw	H'FF'			; if carry set set w at FF
; compare
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry clear then ADRESH< switch setting plus hysteresis so switch reading is within positive hysteresis range
	goto	SW0			; ADRESH > hysteresis range	
; take hysteresis from SWC(x)ls
	movf	HYSTERESIS,w
	subwf	SWC10ls,w
	btfss	STATUS,C
	clrw					; not below 0
	subwf	ADRESH,w
	btfsc	STATUS,C		; if carry is set then ADRESH> switch setting so switch reading is within negative hysteresis
	goto	SW10			; switch found

SW0	; resting position
; no more switches
	clrf		SWITCH			; switch pressed flag
	clrf		RPT				; repeat flag
	clrf		AD_STORE		; last SWC read storage
	goto	READ_CYCLE

SW1
	movf	COMMAND1ls,w
	goto	SW_GENERIC
SW2
	movf	COMMAND2ls,w
	goto	SW_GENERIC
SW3
	movf	COMMAND3ls,w
	goto	SW_GENERIC
SW4
	movf	COMMAND4ls,w
	goto	SW_GENERIC
SW5
	movf	COMMAND5ls,w
	goto	SW_GENERIC
SW6
	movf	COMMAND6ls,w
	goto	SW_GENERIC
SW7
	movf	COMMAND7ls,w
	goto	SW_GENERIC
SW8
	movf	COMMAND8ls,w
	goto	SW_GENERIC

SW9
; switch 9 only runs code once. There is no repeat code
; if RPT,0 is set, bypass running code again
	btfsc	RPT,0
	goto	READ_CYCLE
; run code once only each switch press
	bsf		RPT,0
	movf	COMMAND9ls,w
	goto	SW_GENERIC
SW10
; switch 10 only runs code once. There is no repeat code
; if RPT,0 is set, bypass running code again
	btfsc	RPT,0
	goto	READ_CYCLE
; run code once only each switch press
	bsf		RPT,0
	movf	COMMAND10ls,w

SW_GENERIC
; common instruction for command ( REMOTE_C) and address REMOTE_A)
	movwf	REMOTE_C
	movf	ADDRESSyls,w	; lower byte address
	movwf	REMOTE_A
;
; check protocol
	movf	PROTOCOLls,w
	andlw	B'00001111'		; look at last 4 bits
	btfsc	STATUS,Z		; if zero then NEC 
	goto	NEC1
	xorlw	B'00000001'		; if a 1 then Sony
	btfsc	STATUS,Z
	goto	SONY1
RC5_1
	call		RC5_CODE_SEND
; set switch flag
	bsf		SWITCH,0		; switch pressed
	goto	READ_CYCLE
SONY1
; get address count
; check protocol
	swapf	PROTOCOLls,w
	andlw	B'00001111'		; look at first ms 4 bits
	btfsc	STATUS,Z		; if zero then five bit address
	goto	FIVE
	xorlw	B'00000001'		; if a 1 then eight bits
	btfsc	STATUS,Z
	goto	EIGHT
; 13 bits
	movlw	D'13'
SONY_L
	movwf	SONY_ADD_L	; address counter
	movf	ADDRESSxls,w
	movwf	REM_A_MS		; ms byte of address
	call		SONY_CODE_SEND

; set switch flag
	bsf		SWITCH,0		; switch pressed
	goto	READ_CYCLE
NEC1
	movf	ADDRESSxls,w
	movwf	REM_A_BAR	; ms byte of address
	comf	REMOTE_C,w
	movwf	REM_C_BAR	; complementary value of command
	call		NEC_CODE_SEND
; set switch flag
	bsf		SWITCH,0		; switch pressed
	goto	READ_CYCLE

FIVE
	movlw	D'5'
	goto	SONY_L			; sets address length

EIGHT
	movlw	D'8'
	goto	SONY_L			; set address length

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; IR subroutines
; see Freescale Semiconductor Application Note AN3053 Rev. 1, 03/2008

RC5_CODE_SEND
; start code of two 1s
; 1 toggle bit (changes each new send (unchanged during repeat))
; 5 address bits, ms bits first
; 6 command bits, ms bits first
; repeat at 114ms
; 1 is 889us off, 889us on
; 0 is 889us on 889us off 
; 36kHz modulation

; check if repeat required
	btfss	SWITCH,0
	goto	RC5_CLR

; if set wait for 114ms (2280 x 50us)
	movf	msCOUNT_MS,w		;
	sublw	H'08'				; 
	movwf	TEMP
	movf	msCOUNT_LS,w
	sublw	H'E7'
	btfss	STATUS,C
	decf	TEMP,f
	btfss	TEMP,7				; if set then 114ms or over
	goto	RC5_CODE_SEND	; wait for 114ms 

; clear 114ms counter
RC5_CLR
	bcf		INTCON,GIE			; stop interrupt
	clrf		msCOUNT_LS
	clrf		msCOUNT_MS
	bsf		INTCON,GIE			; restart interrupt

; start bits send two 1's
	call		SEND_1_RC5
	call		SEND_1_RC5

; Toggle. Change toggle bit if a new switch press
	btfsc	SWITCH,0
	incf		TOGGLE,f		; bit 0 toggled 0-1 or 1-0
; send a 1 or 0 as per toggle bit setting
	btfss	TOGGLE,0
	goto	SET_TOGG
	call		SEND_0_RC5 
	goto	SEND_ADD_RC5
SET_TOGG
	call		SEND_1_RC5

SEND_ADD_RC5 
	movlw	D'5'
	movwf	BIT_COUNT	
;
; send address 
; get 5-bits to left side first by shifting left by 3 places
	bcf		STATUS,C
	rlf		REMOTE_A,f	
	bcf		STATUS,C
	rlf		REMOTE_A,f	
	bcf		STATUS,C
	rlf		REMOTE_A,f	
; shift out address
ADDR_LOOP_RC5
	rlf		REMOTE_A,f
	btfss	STATUS,C		; if carry set, send a '1'
	goto	S0
	call		SEND_1_RC5
L1	decfsz	BIT_COUNT,f
	goto	ADDR_LOOP_RC5
	goto	SEND_COMM_RC5
S0	call		SEND_0_RC5
	goto	L1

SEND_COMM_RC5
	movlw	D'6'
	movwf	BIT_COUNT	

; send command codes
; get 6-bits to left side first by shifting left by 2 places
	bcf		STATUS,C
	rlf		REMOTE_C,f	
	bcf		STATUS,C
	rlf		REMOTE_C,f	

; shift out  command 
COMM_LOOP_RC5
	rlf		REMOTE_C,f
	btfss	STATUS,C		; if carry set send a '1'
	goto	S01
	call		SEND_1_RC5
L11	decfsz	BIT_COUNT,f
	goto	COMM_LOOP_RC5
	goto	CHK_114
S01	call		SEND_0_RC5
	goto	L11
CHK_114
; end sequence so outputs off
	bcf		GPIO,0			; demodulated output off
; 36kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	return						;

SEND_0_RC5
; 889us on

	bsf		GPIO,0			; demodulated output on
; 36kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_889us

; 889us off
	bcf		GPIO,0			; demodulated output off
; 36kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_889us
	return

SEND_1_RC5

;889us off
	bcf		GPIO,0			; demodulated output off
; 36kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_889us
; 889us on
	bsf		GPIO,0			; demodulated output on
; 36kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_889us
	return


; 889us timer
T_889us
	clrf		TMR1L
	clrf		TMR1H
	
; Timer1 increments at 1us rate. so 889 x 1us (H379 x 1us is 889us)
T_889
	movlw	B'00000011'
	xorwf	TMR1H,w		; test for a 3
	btfss	STATUS,Z
	goto	T_889
	movf	TMR1L,w
	sublw	H'70'			;  less  than H79 to compensate for cycles in code 
	btfsc	STATUS,C
	goto	T_889
	return
;...............................................................................................................................................................................................................

SONY_CODE_SEND
; Sony protocol
; Pulse width protocol, known also as SIRC
; There are 12 bit, 15 and 20 bits versions. The data consists of 7 command bits sent first and remaining are address. 
; data is started with a 2.4 ms mark, and a 0.6 ms space 
; ls bits sent first
; 1 is  a 1200 s mark period followed by a 600 s space period
; 0 is  a 600 s mark period followed by a 600 s space period
; Carrier frequency is 40 kHz.
; repeat  is by sending the data again including the 2.4ms mark and 0.6ms space. Repeated every 45 ms
; when reading, count the address bits used

; check if repeat required
	btfss	SWITCH,0
	goto	SONY_CLR
; if set wait for 45ms (900 x 50us)
	movf	msCOUNT_MS,w		;
	sublw	H'03'				; 
	movwf	TEMP
	movf	msCOUNT_LS,w
	sublw	H'83'
	btfss	STATUS,C
	decf	TEMP,f
	btfss	TEMP,7				; if set then 45ms or over
	goto	SONY_CODE_SEND	; wait for 45ms 

; clear 45ms counter
SONY_CLR
	bcf		INTCON,GIE			; stop interrupt
	clrf		msCOUNT_LS
	clrf		msCOUNT_MS
	bsf		INTCON,GIE			; restart interrupt
; ON
;  2.4ms burst
	bsf		GPIO,0			; demodulated output on
; 40kHz modulated	
	movlw	D'25'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; send for count of D48 in ms counter (48 x 50us=2.4ms)
SEND1S
	movf	msCOUNT_LS,w
	sublw	D'47'			; 1 less
	btfsc	STATUS,C
	goto	SEND1S

; OFF
; off for 0.6ms
	bcf		GPIO,0			; demodulated output off; 40kHz modulated off	
	movlw	D'00'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; D60 for 2.4ms and 0.6ms total count in ms counter
SEND2S
	movf	msCOUNT_LS,w	; 
	sublw	D'59'
	btfsc	STATUS,C
	goto	SEND2S

; send out 7-command bits

SEND_COM_SONY
	movlw	D'7'
	movwf	BIT_COUNT	

; send command codes
; shift out command
COMM_LOOP_SONY
	rrf		REMOTE_C,f
	btfsc	STATUS,C		; if carry set send a '1'
	goto	S0S
	call		SEND_0_SONY
L1S	decfsz	BIT_COUNT,f
	goto	COMM_LOOP_SONY
	goto	SEND_ADD_SONY
S0S	call		SEND_1_SONY
	goto	L1S

SEND_ADD_SONY

	movf	SONY_ADD_L,w	; sony address length 5, 8 and 13 bits (for 12, 15 and 20 bits total including command bits (7))
	movwf	BIT_COUNT	

; send address  codes
; shift out address
ADD_LOOP_SONY
	rrf		REM_A_MS,f
	rrf		REMOTE_A,f
	btfsc	STATUS,C		; if carry set send a '1'
	goto	S01S
	call		SEND_0_SONY
L11S
	decfsz	BIT_COUNT,f
	goto	ADD_LOOP_SONY
	goto	CHK_45
S01S
	call		SEND_1_SONY
	goto	L11S
CHK_45
; end sequence so outputs off
	bcf		GPIO,0			; demodulated output off
; 40kHz modulated off	
	movlw	D'00'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	return					

; a zero 
SEND_0_SONY
; 600us on

	bsf		GPIO,0			; demodulated output on
; 40kHz modulated	
	movlw	D'25'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_600us

; 600us off
	bcf		GPIO,0			; demodulated output off
; 40kHz modulated off	
	movlw	D'00'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_600us
	return

; a 1
SEND_1_SONY
 ; 1200us on
	bsf		GPIO,0			; demodulated output on
; 40kHz modulated	
	movlw	D'25'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_600us
	call		T_600us
;600us off
	bcf		GPIO,0			; demodulated output off
; 40kHz modulated off	
	movlw	D'00'			; zero for off and D25 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_600us
	return

; 600us timer
T_600us
	clrf		TMR1L
	clrf		TMR1H
; Timer1 increments at 1us rate. so 600 x 1us (H258 x 1us is 600us)
T_600
	movlw	B'00000010'
	xorwf	TMR1H,w		; test for a 2
	btfss	STATUS,Z
	goto	T_600
	movf	TMR1L,w
	sublw	H'50'			;  less  than H58 to compensate for cycles in code 
	btfsc	STATUS,C
	goto	T_600
	return

;.............................................................................................................................................................................................................

NEC_CODE_SEND 
; NEC protocol
; IR signal that's based on pulse distance protocol
; start is a 38kHz burst at 9ms and 4.5ms of no burst
; data sent as an 8-bit address (LSB first) then a complementary version of the address or an address extension
; further data sent as an 8-bit command (LSB first) then a complementary version of the command
; logic 1 is 560us 38kHz burst and 1690us of no burst
; logic 0 is a 560us 38kHz burst and 560us of no burst
; At the end of the 32bit data is a tail pulse comprising 560us of a 38kHz burst
; a repeat indication (transmit button held on) is 9000us 38kHz burst followed by 2250us no burst and a 560us 38kHz burst
; this is repeated each 110ms
	btfsc	SWITCH,0		; if set then repeat	
	goto	REPEAT_NEC

; clear 110ms counter
	bcf		INTCON,GIE			; stop interrupt
	clrf		msCOUNT_LS
	clrf		msCOUNT_MS
	bsf		INTCON,GIE			; restart interrupt
; ON
;  9ms burst
	bsf		GPIO,0			; demodulated output on
; 38kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; send for count of D180 in ms counter (180 x 50us=9ms)
SEND1C
	movf	msCOUNT_LS,w
	sublw	D'179'			; 1 less
	btfsc	STATUS,C
	goto	SEND1C

; OFF
; off for 4.5ms
	bcf		GPIO,0			; demodulated output off
; 38kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; D270 for 9ms and 4.5ms total count in ms counter (270 is H10E) 
SEND2C
	btfss	msCOUNT_MS,0	; if set check for >0D
	goto	SEND2C	
	movf	msCOUNT_LS,w
	sublw	H'0D'			; 1 less
	btfsc	STATUS,C
	goto	SEND2C

	movlw	D'32'
	movwf	BIT_COUNT		; 32 bit counter

; send address and command codes
; shift out address, inverted address, command and inverted command
DATA_LOOP
	rrf		REM_C_BAR,f
	rrf		REMOTE_C,f
	rrf		REM_A_BAR,f
	rrf		REMOTE_A,f
	btfss	STATUS,C		; if carry set send a '1'
	goto	SEND_0

SEND_1
; 560us on

	bsf		GPIO,0			; demodulated output on
; 38kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_560us

; 1690us off

	bcf		GPIO,0			; demodulated output off
; 38kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_1690us

	decfsz	BIT_COUNT,f
	goto	DATA_LOOP
	goto	TAIL

SEND_0
;560us on
;560us off

	bsf		GPIO,0			; demodulated output on
; 38kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_560us

; 560us off

	bcf		GPIO,0			; demodulated output off
; 38kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_560us

	decfsz	BIT_COUNT,f
	goto	DATA_LOOP
	goto	TAIL

TAIL
	;560us off

	bsf		GPIO,0			; demodulated output on
; 38kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	call		T_560us

; off

	bcf		GPIO,0			; demodulated output off
; 38kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM
	return					; end of data


; 560us timer
T_560us
	clrf		TMR1L
	clrf		TMR1H
; Timer1 increments at 1us rate. so 560 x 1us (H230 x 1us is 560us)
T_560
	btfss	TMR1H,1		; test for the H2, bit 1 set 
	goto	T_560
	movf	TMR1L,w
	sublw	H'27'			;  less  than H30 to compensate for cycles in code 
	btfsc	STATUS,C
	goto	T_560
	return

; 1690us timer
T_1690us
	clrf		TMR1L
	clrf		TMR1H
; Timer1 increments at 1us rate so  D1690 (H69A) x 1us is 1690 
T_1690
	btfss	TMR1H,1		; test for the H6, bits 1 and 2 set 
	goto	T_1690
	btfss	TMR1H,2
	goto	T_1690
	movf	TMR1L,w
	sublw	H'96'			;   less than H9A to compensate for cycles in code
	btfsc	STATUS,C
	goto	T_1690
	nop						; nops added to get timer period correct	
	nop
	return

;------------------------------------------------------------------------
REPEAT_NEC

; REPEAT signal is a 9ms burst followed by 2250us off and 560us burst repeated 110ms apart
; wait for 110ms. That is 50us each count so count is D2200, H898

; check for 110ms time period 
	movf	msCOUNT_MS,w		;
	sublw	H'08'				; 
	movwf	TEMP
	movf	msCOUNT_LS,w
	sublw	H'97'
	btfss	STATUS,C
	decf	TEMP,f
	btfss	TEMP,7				; if set then 110ms or over
	goto	REPEAT_NEC		; wait for 110ms 
; 110ms counter restarts from 0
	bcf		INTCON,GIE			; stop interrupt
	clrf		msCOUNT_LS
	clrf		msCOUNT_MS
	bsf		INTCON,GIE			; restart interrupt	
	
; ON
;  9ms burst
	bsf		GPIO,0			; demodulated output on
; 38kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; send for count of D180 in ms counter (180 x 50us=9ms)
SEND1
	movf	msCOUNT_LS,w
	sublw	D'179'			; 1 less
	btfsc	STATUS,C
	goto	SEND1

; OFF
; off for 2250us
	bcf		GPIO,0			; demodulated output off
; 38kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; send for count of D225 in ms counter (45 x 50us=2250us added to the original 9ms count of 180)
SEND2
	movf	msCOUNT_LS,w
	sublw	D'224'			; 1 less
	btfsc	STATUS,C
	goto	SEND2

; ON
; 560us on
	bsf		GPIO,0			; demodulated output on
; 38kHz modulated	
	movlw	D'26'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM

; send for count of 560us. 	
	call		T_560us

; OFF
	bcf		GPIO,0			; demodulated output off
; 38kHz modulated off	
	movlw	D'00'			; zero for off and D26 for square wave on
	movwf	CCPR1L		; ms byte of PWM

	return

; *****************************************************************************************************************************************************
; Read IR codes

; IR subroutines

START ; start of Infra Red reception (usage: Call START if returns with 00 or FF then not valid. A0 for A ok is valid code)
; wait for start with GPIO,3 low
	bcf		INTCON,GIE		; stop interrupt
START1
	btfsc	GPIO,3
	goto	START1	
		
; Clear timer1
	clrf		TMR1L			; 1us counter
	clrf		TMR1H
	bsf		T1CON,0		; start timer

; Timer1 increments at 1us rate. 

; wait for a high at GPIO,3. when high, test the time period
W1	btfss	GPIO,3
	goto	W1

; NEC is 9ms low (modulated signal received)
; Sony is 2.4ms low
; RC5 is 889us low

; test for RC5 ; test for 889us. Range allows 10% at between 800us and 977us
	bcf		T1CON,0	; stop timer and read
;  is it >800us
	movf	TMR1H,w		;
	sublw	H'3'			 
	movwf	TEMP
	movf	TMR1L,w
	sublw	H'20'
	btfss	STATUS,C
	decf	TEMP,f
	btfss	TEMP,7		
	retlw	D'00'			; less than 800us so not an IR code
; is it <977us
	movf	TMR1H,w		;
	sublw	H'03'			 
	movwf	TEMP
	movf	TMR1L,w
	sublw	H'D1'
	btfss	STATUS,C
	decf	TEMP,f
	btfss	TEMP,7		
	goto	RC5_READ
; test for NEC
; is it >8ms (D8000, H1F40)
	movf	TMR1H,w		;
	sublw	H'1F'			 
	movwf	TEMP
	movf	TMR1L,w
	sublw	H'3F'
	btfss	STATUS,C
	decf	TEMP,f
	btfsc	TEMP,7		
	goto	NEC_READ	; NEC code
; test for Sony
; is it >2.2ms (2200 or H'898)
	movf	TMR1H,w		;
	sublw	H'08'			
	movwf	TEMP
	movf	TMR1L,w
	sublw	H'98'
	btfss	STATUS,C
	decf	TEMP,f
	btfsc	TEMP,7		
	goto	SONY_READ
	retlw	D'00'			; not recognised
NEC_READ

; read IR signal that's based on pulse distance protocol
; start train is a 38kHz burst at 9ms and 4.5ms of no burst
; data sent as an 8-bit address (LSB first) then a complementary version of the address or address extension
; further data sent as an 8-bit command (LSB first) then a complementary version of the command
; logic 1 is 560us 38kHz burst and 1690us of no burst
; logic 0 is a 560us 38kHz burst and 560us of no burst
; At the end of the 32bit data is a tail pulse comprising 560us of a 38kHz burst
; data is sent once
; a repeat indication (transmit button held on) is 9000us 38kHz burst followed by 2250us no burst and a 560us 38kHz burst
; this is repeated each 110ms
; Check for start train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 4.5ms (no 38kHz signal)

	bsf		INTCON,GIE		; allow interrupt
; is it <10ms (D10000, H2710)
	movf	TMR1H,w	; ms
	sublw	H'27'				
	movwf	TEMP
	movf	TMR1L,w
	sublw	H'10'
	btfss	STATUS,C
	decf	TEMP,f
	btfsc	TEMP,7		
	retlw	D'00'		; no valid IR code
; GPIO,3 low for ~9ms (8-10ms) continue with NEC IR code decoding

; GPIO,3 has stayed low and then gone high within the acceptable ~9ms low period (8-11ms window)

; Now check for 4.5ms high (3.5 to 4.5ms window)
	movlw	D'70'		; 3.5ms
	movwf	COUNTER

; check GPIO,3
CK_3_2
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_2 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_2

; GPIO,3 can now go low up to a period of 2ms
; wait for more interrupts
	movlw	D'40'		; 2ms
	movwf	COUNTER
; GPIO,3 can go low in this period
CK_3_3
	btfss	GPIO,3
	goto	FOUR_POINT_FIVE_ms	; GPIO,3 has gone low within the acceptable ~4.5ms high period (3.5-5.5ms window)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_3
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_3
	retlw	H'00'		; GPIO,3 stays high

; end of start train check
FOUR_POINT_FIVE_ms
	movlw	D'32'		; set up 32bit counter
	movwf	BIT_COUNT

; START_DATA ; start of data 
; clear data
	clrf	REMOTE_A		; remote control Address byte
	clrf	REM_A_BAR	; complementary value of address or address extension
	clrf	REMOTE_C		; remote control Command byte
	clrf	REM_C_BAR	; complementary value of command

; start reading data
READ_DATA_NEC
; a 560us low
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3
CK_3_4
	btfsc	GPIO,3
	retlw 	H'00'		; if GPIO,3 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_4 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_4
; GPIO,3 can now go high up to a period of 400us for a 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_5
	btfsc	GPIO,3
	goto	ONE_ZERO	; GPIO,3 has gone high within the acceptable ~560us low period (400us-800us window)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_5
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_5
	retlw 	H'FF' 		; GPIO,3 stays low  
; test for a zero (high for 560us) or a one (high for 1690us)
ONE_ZERO
; Check for 560us high
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3

CK_3_6
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_6 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_6

; GPIO,3 can now go high up to a period of 400us for a 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER

; GPIO,3 can go low in this period
CK_3_7
	btfss	GPIO,3
	goto	LOGIC_0	; GPIO,3 has gone low within the acceptable ~560us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_7
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_7
	goto	CHECK_FOR_1

LOGIC_0	; logic zero detected
	bcf		STATUS,C	; clear carry
; shift detected data into storage
SHIFT_DATA_NEC	
	rrf		REM_C_BAR,f	; complementary value of command
	rrf		REMOTE_C,f	; remote control Command byte
	rrf		REM_A_BAR,f	; complementary value of address or address extension
	rrf		REMOTE_A,f		; remote control Address byte
	
	decfsz	BIT_COUNT,f		; shift data for 32 bits
	goto	READ_DATA_NEC	; contine reading data
	goto	TAIL_NEC_RD	; end of 32 bit data, check for the tail (560us of burst)
	
CHECK_FOR_1 ; check for a logic 1

; Check for 1690us high  (already 560us high) so check for 1130us
	movlw	D'10'		; 500us (1300us min)
	movwf	COUNTER
; check GPIO,3

CK_3_8
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_8 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_8

; GPIO,3 can now go high up to a period of 600us extra
; wait for more interrupts

	movlw	D'14'		; 700us (2000us max)
	movwf	COUNTER

; GPIO,3 can go low in this period
CK_3_9
	btfss	GPIO,3
	goto	LOGIC_1	; GPIO,3 has gone low within the acceptable ~1690us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_9
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_9
	retlw	H'00'		; error	

LOGIC_1
	bsf		STATUS,C	; data is a 1
	goto	SHIFT_DATA_NEC

TAIL_NEC_RD ; end of data tail

; a 560us low
	movlw	D'8'			; allow a 400us minimum 
	movwf	COUNTER
; check GPIO,3
CK_3_10
	btfsc	GPIO,3
	retlw 	H'00'		; if GPIO,3 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_10 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_10
; GPIO,3 can now go high up to a period of 400us for a 800us maximum
; wait for more interrupts
	movlw	D'8'			; 400us
	movwf	COUNTER
; GPIO,3 can go high in this period
CK_3_11
	btfsc	GPIO,3
	goto	TAIL_DET	; GPIO,3 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_3_11
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_3_11
	retlw 	H'FF' 		; error

TAIL_DET ; tail is detected

;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
; check if Address and Address bar are complementary
; *****not included to allow REM_A_BAR to be an address exension rather than a complementary REMOTE_A
;	comf	REM_A_BAR,w	; complementary value of address
;	xorwf	REMOTE_A,w	; remote control Address byte
;	btfss	STATUS,Z
;	retlw	H'00'			; error
;>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

; check if Command and Command bar are complementary

	comf	REM_C_BAR,w	; complementary value of command
	xorwf	REMOTE_C,w	; remote control Command byte
	btfss	STATUS,Z
	retlw	H'FF'			; error

	clrf		PROTOCOLls	; set for NEC protocol
	retlw	H'A0'			; if data is decoded correctly send an 'AO'k return message

;.........................................................................................................................................................................................

SONY_READ
; There are 12 bit, 15 and 20 bits versions. The data consists of 7 command bits sent first and remaining bits are address. so 5, 8 or 13 bit address
; The data is started with a 2.4 ms mark, and a 0.6 ms space 
; 1 is  a 1200 s mark period followed by a 600 s space period
; 0 is  a 600 s mark period followed by a 600 s space period
; Carrier frequency is 40 kHz.
; repeat is by sending the data again including the 2.4ms mark and 0.6ms space. Repeated every 45 ms
; when reading, count the address bits used

; is it <2.7ms (D2700, H)
	bcf		T1CON,0	; stop timer and read
	movf	TMR1H,w	; ms
	sublw	H'A'				
	movwf	TEMP
	movf	TMR1L,w
	sublw	H'8C'
	btfss	STATUS,C
	decf	TEMP,f
	btfsc	TEMP,7		
	retlw	D'00'		; no valid IR code

; GPIO,3 low for ~2.4ms continue with Sony IR code decoding
	bsf		INTCON,GIE		; allow interrupt

; Now check for 0.6ms high (500us to 700us window)
	movlw	D'10'		; 500us
	movwf	COUNTER
; check GPIO,3

SYCK_3_2
	btfss	GPIO,3
	retlw	H'FF'		; if GPIO,3 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	SYCK_3_2 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	SYCK_3_2

; GPIO,3 can now go low up to a period of 200us to add up to 700us
; wait for more interrupts
	movlw	D'4'		; 200us
	movwf	COUNTER
; GPIO,3 can go low in this period
SYCK_3_3
	btfss	GPIO,3
	goto	EOS_SONY	; GPIO,3 has gone low within the acceptable ~600 high period (500us to 700us window)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	SYCK_3_3
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	SYCK_3_3
	retlw	H'00'		; error return

; end of start train check
EOS_SONY ; end of start train for Sony code
	movlw	D'22'		; set up 24 bit counter to cover the 7 bit command and 5, 8 or 13bit address total length of 20maximum address and command  
	movwf	BIT_COUNT

; START_DATA ; start of data 
; clear data
	clrf	REMOTE_A		; remote control Address byte
	clrf	REM_A_MS		; address ms byte
	clrf	REMOTE_C		; remote control Command byte
	clrf	SONY_ADD_L	; address counter 12, 15 or 20 bit data (5, 8 or 13 bit address)

; start reading data
READ_DATA_SONY
; a 600us low is a '0, a 1200us low is a '1'
	clrf		COUNTER	; start at zero and count up
; check GPIO,3
; check when GPIO,3 goes high
SYCK_3_4
	btfsc	GPIO,3
	goto	SY_3
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto  	SYCK_3_4 
	bcf		INT_FLG,0
	incfsz	COUNTER,f	; count interrupts
	goto	CK_CNTR_SONY
	retlw	H'FF'		; data error as over in length. (unlikely to be low that long) see bit 5 check that detects over in length
; check counter for > 32 if bit 5 is set. That's a low of 1600us  

CK_CNTR_SONY
	btfss	COUNTER,5	; bit 5 check for over a count of 32
	goto	SYCK_3_4
	retlw	H'FF'		; data read error	
SY_3
	
; check how long GPIO,3 was low
; COUNTER should be between 50us x 10 to 14 for a '0' (~600us)
; > a 10
	movf	COUNTER,w
	sublw	D'9'				; if 10 or more check for less than 14
	btfsc	STATUS,C
	retlw	D'00'			; error return as <10
; < 14
	movf	COUNTER,w
	sublw	D'14'			; if 10 or more check for less than 14
	btfss	STATUS,C
	goto	SONY_1			; check for a '1'
; detected as a '0'
	bcf		STATUS,C		; clear carry for a '0'
	goto	SHIFT_DATA_SONY 

SONY_1 ; check for a '1'
; COUNTER should be 50us x 22 to 26 for a '1' (1200us)
; > a 22
	movf	COUNTER,w
	sublw	D'21'			; if 22 or more check for less than 26
	btfsc	STATUS,C
	retlw	D'00'			; error return as <10
; < 26
	movf	COUNTER,w
	sublw	D'26'			; if 22 or more check for less than 26
	btfss	STATUS,C
	retlw	D'00'			; error return
; detected as a '1'
	bsf		STATUS,C		; set carry for a '1'

; shift detected data into storage
SHIFT_DATA_SONY

	rrf		REM_A_MS,f	; ms remote address byte
	rrf		REMOTE_A,f		; remote control Address byte	
	rrf		REMOTE_C,f	; remote control Command byte	; (this ends up having bit 7 with address ls byte that is fixed later) 

; wait for the time GPIO,3 is high. If over 600us then stop increasing SONY_ADD_L (address length counter)
; shift data with zeroes until BIT_COUNT is zero 

	movlw	D'14'		; 700us
	movwf	COUNTER

; GPIO,3 can go low in this period
SYCK_3_6
	btfss	GPIO,3
	goto	SONY_6		; GPIO,3 has gone low within the acceptable ~600 high period (up to 700us)
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	SYCK_3_6
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	SYCK_3_6
	goto	END_DATA_SONY; end of data as GPIO,3 remains high >700us

SONY_6	
	incf		SONY_ADD_L,f	; address counter 12, 15 or 20 bit data (for 5, 8 or 13 bit address)
	decfsz	BIT_COUNT,f		; shift data for 24 bits
	goto	READ_DATA_SONY	; contine reading data
	retlw	H'00'			; bit count at zero so error. Should have stopped at 20 maximum rather than 24
	
END_DATA_SONY
	
	bcf		STATUS,C
	rrf		REM_A_MS,f	; ms remote address byte
	rrf		REMOTE_A,f		; remote control Address byte	
	rrf		REMOTE_C,f	; remote control Command byte	; (this ends up having bit 7 with address ls byte that is fixed later) 
	decfsz	BIT_COUNT,f		; shift data for 24 bits
	goto	END_DATA_SONY; continue to shift untill bit count is zero
	

; fix command and address shifts, shift command (for 24bit shift)
; move command right as is 7-bits not 8
	bcf		STATUS,C
	rrf		REMOTE_C,f
; set protocol
	movlw	D'01'
	movwf	PROTOCOLls
; get address length
	incf		SONY_ADD_L,f
	movf	SONY_ADD_L,w	; address counter 12, 15 or 20 bit data (5, 8 or 13 bit address)
	xorlw	D'12'
	btfsc	STATUS,Z
	retlw	H'A0'			; return with A 0k message, code read successfully (ms its of PROTOCOLls left as 00
	movf	SONY_ADD_L,w	; address counter 12, 15 or 20 bit data (5, 8 or 13 bit address)
	xorlw	D'15'
	btfsc	STATUS,Z
	goto	SET_8
	movf	SONY_ADD_L,w	; address counter 12, 15 or 20 bit data (5, 8 or 13 bit address)
	xorlw	D'20'
	btfsc	STATUS,Z
	goto	SET_13
	retlw	D'00'			; error
SET_8
	movlw	H'11'			; 1 for 8-bit address (high nibble), 1 for Sony protocol (low nibble)
	movwf	PROTOCOLls
	retlw	H'A0'

SET_13
	movlw	H'21'			; 2 for 13-bit address (high nibble), 1 for Sony protocol (low nibble)
	movwf	PROTOCOLls
	retlw	H'A0'
;........................................................................................................................................................................................
RC5_READ		
; 
	bsf		T1CON,0	; run timer and read
; start code of two 1s
; 1 toggle bit (changes each new send (unchanged during repeat))
; 5 address bits, ms bits first
; 6 command bits, ms bits first
; repeat at 114ms
; 1 is 889us off, 889us on
; 0 is 889us on 889us off 
; most significant bits sent/received first
  
; use timer1 to count between two lows for 2 x 889us or 1778us. Use as the clock to get data
; GPIO,3 high so wait for low and use TMR1H and TMR1L as data rate

; GPIO,3 just gone high
; wait for a low again
RC5_L
	btfsc	GPIO,3
	goto	RC5_L

; timer is 1.778ms. Wait for 1.75 x timer before a data read. Timer value x 1 for consecutive data reads. '1' is a high, '0' is a low at each read point
; shift left for 11
	bcf		T1CON,0	; stop timer and read
	movf	TMR1H,w	; ms
	movwf	T1_STO_H	; timer1 store ms byte
	movf	TMR1L,w	; ms
	movwf	T1_STO_L	; timer1 store ms byte

;
; calculate 1/2 and 1/4 of timer value
	bcf		STATUS,C
	rrf		T1_STO_H,w
	movwf	TEMPM
	rrf		T1_STO_L,w
	movwf	TEMPL		; /2

	bcf		STATUS,C
	rrf		TEMPM,w
	movwf	TEMP2M
	rrf		TEMPL,w	; /4
	movwf	TEMP2L

; add 1/2 and 1/4 to timer to get  1.75 times timer
	
	movf	T1_STO_H,w	; ms byte of counter
	addwf	TEMPM,f		; add 1/2 count ms byte
	movf	T1_STO_L,w
	addwf	TEMPL,f			; 1/2 count ls byte
	btfsc	STATUS,C		; if carry increase ms byte	
	incf		TEMPM,f

	movf	TEMPM,w		; ms byte of 1 +1/2 counter
	addwf	TEMP2M,f		; add 1/4 count ms byte
	movf	TEMPL,w
	addwf	TEMP2L,f		; 1/4 count ls byte
	btfsc	STATUS,C		; if carry increase ms byte	
	incf		TEMP2M,f

; set up timer 1 for overflow flag
	clrf		TMR1L 			; Clear Low byte
	bcf		T1CON,TMR1ON	; timer off
	movf	TEMP2L,w		; low byte
	sublw	H'FF'			; low byte subtract
	movwf	TMR1L			; timer 1 low byte
	movf	TEMP2M,w		; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	TMR1H
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
; (19us out due to above code)

; START_DATA ; start of data 
; clear data

	clrf		REMOTE_A		; remote control Address byte 5-bits
	clrf		REMOTE_C		; remote control Command byte 6 bits
	movlw	D'11'
	movwf	BIT_COUNT
	goto	WAIT_RC5_1

; start reading data at 1.778ms intervals
READ_RC5
; each time reduce T1_STO_H and T1_STO_L  by x to compensate the loss in time when reloading timer1
	movlw	D'1'
	addwf	T1_STO_L,f
	movlw	D'00'
	btfsc   	STATUS,C    	 	; carry check
 	addlw   	D'1'       		 	; add one if carry
	addwf	T1_STO_H,f

; set up timer 1 for overflow flag. T1_STO_H, T1_STO_L have the nominal 1.778ms as measured in the two start bits. So uses the transmitter data rate for clocking 

	bcf		T1CON,TMR1ON	; timer off
	movf	T1_STO_L,w		; low byte
	sublw	H'FF'			; low byte subtract
	movwf	TMR1L			; timer 1 low byte
	movf	T1_STO_H,w		; high byte
	btfss   	STATUS,C    	 	; carry check
    	addlw   	D'1'       		 	; add one if carry
	sublw	H'FF'			; w has high byte subtraction
	movwf	TMR1H
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared
	bsf		T1CON,TMR1ON	; timer on
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag cleared

; wait for overflow flag 
WAIT_RC5_1	
	btfss	PIR1,TMR1IF		; timer 1 interrupt flag cleared
	goto	WAIT_RC5_1
; read data
; set or clear carry and shift into command and address
	bcf		STATUS,C
	btfsc	GPIO,3
	bsf		STATUS,C
; shift detected data into storage
;11 shifts
SHIFT_DATA_RC5

	rlf		REMOTE_C,f	; remote control Command byte	; 
	rlf		REMOTE_A,f		; remote control Address byte	

	decfsz	BIT_COUNT,f		; shift data for 11 bits
	goto	READ_RC5
;(this ends up having address bits in command; so shift left twice more for address. Then shift command right with a cleared carry.)
	bcf		STATUS,C
	rlf		REMOTE_C,f	; remote control Command byte	; 
	rlf		REMOTE_A,f		; remote control Address byte	
	bcf		STATUS,C
	rlf		REMOTE_C,f	; remote control Command byte	; 
	rlf		REMOTE_A,f		; remote control Address byte	
	bcf		STATUS,C
	rrf		REMOTE_C,f
	bcf		STATUS,C
	rrf		REMOTE_C,f	

	movlw	D'02'			; set protocol for RC5
	movwf	PROTOCOLls
	retlw	H'A0'			; return with A 0k message, code read successfully



; *****************************************************************************************************************************************************
; subroutine to read hysteresis and then restore to read SWC input

HYSTERESIS_RUN
; read hysteresis (AN1) set for 0-500mV range 
; acknowledge with LED1 on
	bsf		GPIO,0	
; bits 4-2
; 001 = Channel 01 (AN1)
	movlw	B'00000100'		; channel 1, left justified, VDD ref etc
	movwf	ADCON0
	bsf		ADCON0,0		; A/D on
	call		DELAY100		;  100ms	
	call		ACQUIRE_AD	; read hysteresis voltage
	movf	ADRESH,w
	movwf	TEMP
; divide by 8
	bcf		STATUS,C
	rrf		TEMP,f
	bcf		STATUS,C
	rrf		TEMP,f
	bcf		STATUS,C
	rrf		TEMP,f
; place in HYSTERESIS
	movf	TEMP,w
	movwf	HYSTERESIS
; divide by a further 4 (for a divide by 32 total)
	bcf		STATUS,C
	rrf		TEMP,f
	bcf		STATUS,C
	rrf		TEMP,f
; subtract from HYSTERESIS to get a nominal divide by 10 (actually 3/32 so accurate enough for ~20mV digitals steps) 
	movf	TEMP,w
	subwf	HYSTERESIS,f

; setup for SWC input read
; bits 4-2
; 011 = Channel 03 (AN3)
	movlw	B'00001100'		; channel 3, left justified, VDD ref etc
	movwf	ADCON0
	bsf		ADCON0,0		; A/D on
; delay	
	call		DELAY100		;  100ms	
	
; acknowledge LED1 off
	bcf		GPIO,0	
	return
;..............................

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,1	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,1	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return
;.................................................

; delay loop 
DELAYms; approx 5ms
	movlw	D'23'		; delay value
DY	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'117'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8
	return

; delay loop 
DELAY100 ; approx 100ms
	movlw	D'255'		; delay value
	movwf	STORE1		; STORE1 is number of loops value
LOOP800	
	movlw	D'213'
	movwf	STORE2		; STORE2 is internal loop value	
LOOP900
	decfsz	STORE2,f
	goto	LOOP900
	decfsz	STORE1,f
	goto	LOOP800
	return
;........................................

; read data memory
READ; 'w' has read data
	bsf		STATUS,RP0	; select memory bank 1 
	movf	READADDMS,w 
	movwf 	PMADRH		; ms Byte of Program Address to read
	movf	READADDLS,w
	movwf 	PMADRL 		; ls Byte of Program Address to read

	bsf	 	PMCON1,RD 	; Read
	nop 					
	nop
; memory is read in second cycle PM read instruction
	movf	PMDATH,w 		; ms Byte of Program data 
;	movwf	TEMP_VAL
	movf	PMDATL,w 		; ls Byte of Program data
	bcf		STATUS,RP0	; bank 0
	return
;.............................................

; write to data memory
WRITE
	bcf		 INTCON,GIE 	; Disable interrupts 
G	btfsc	 INTCON,GIE		 ; See AN576
	goto	 G

	bsf		STATUS,RP0	; select memory bank 1
	movf	READADDMS,w 
	movwf 	PMADRH		; MS Byte of Program Address to write
	movf	READADDLS,w 
	movwf 	PMADRL 		; LS Byte of Program Address to write

	movf	DATA_ADD,w	; Load initial data address (must have start at a 16byte boundary 
	movwf	FSR 
LOOP_WRITE 
	movf	 INDF,w 			; Load first data byte into upper byte
	movwf	 PMDATH		;
	incf		 FSR,f 			; Next byte
	movf	 INDF,w 			; Load second data byte into lower byte
	movwf	 PMDATL 		; 		
	incf		 FSR,f 			;
	bsf		 PMCON1,WREN ; Enable writes

; Required Sequence
	movlw	H'55'			; Start of required write sequence:
	movwf	PMCON2		; Write 55h
	movlw	H'AA'		 	;
	movwf	PMCON2 		; Write AAh
	bsf		PMCON1,WR 	; Set WR bit to begin write
	nop					 	; NOPs required for time to transfer data to the buffer registers
	nop						; 
;
	bcf		PMCON1,WREN ; Disable writes
	movf	PMADRL,w
	incf		PMADRL,f		; Increment address
; select 4,8,12 or16 	
	; 0F = 16 words
	; 0B = 12 words
	; 07 =  8 words
	; 03 =  4 words
	andlw	H'0F'			; Indicates when the set number of words have been programmed
	sublw	H'0F'

	btfss	STATUS,Z 		; Exit on a match,
	goto	LOOP_WRITE	; Continue until ended
	bcf		STATUS,RP0	; bank 0
	bsf		 INTCON,GIE 	; enable interrupts 
	return

	end
